#include <windows.h>
#include <vector>
#include <string>
#include <string.h>
#include "IntrFace\CDDraw.h"
#include "Graphics\PrimDraw.h"
#include "Vectors\VectObject.h"
#include "Binfile\BinfStd.h"
#include "Externs.h"
#include "Resource.h"
#include "Editor.h"

static std::vector<Vectors::VectObject *> Objects;
static int ObjIndex;

static CDDraw Cddraw;
static Graphics::PrimDraw PD;
static CDDSurface *BackBuffer;
static int DispWid, DispHei, DispX, DispY;

struct Item
{
   int Min[NUMATTRS], Max[NUMATTRS], Avg[NUMATTRS];
   int Cursed, Weight;
   int Object, Type, Place;

   std::string ItemName, EnumName;
   Item();
   void Save(binfile *file);
   void Load(binfile *file);
};
Item::Item()
{
   for(int i=0;i<NUMATTRS;i++) Min[i] = Max[i] = Avg[i] = 0;
   Cursed = Weight = Object = Type = 0;
}
void Item::Save(binfile *file)
{
   file->write(&Type,   sizeof(int));
   file->write(&Object, sizeof(int));
   file->write(&Place,  sizeof(int));
   file->write(&Cursed, sizeof(int));
   file->write(&Weight, sizeof(int));
   for(int i=0;i<NUMATTRS;i++)
   {
      file->write(&Min[i], sizeof(int));
      file->write(&Max[i], sizeof(int));
      file->write(&Avg[i], sizeof(int));
   }
   int len = ItemName.size() + 1;
   file->write(&len, sizeof(int));
   file->write(ItemName.c_str(), len);
}
void Item::Load(binfile *file)
{
   file->read(&Type,   sizeof(int));
   file->read(&Object, sizeof(int));
   file->read(&Place,  sizeof(int));
   file->read(&Cursed, sizeof(int));
   file->read(&Weight, sizeof(int));
   for(int i=0;i<NUMATTRS;i++)
   {
      file->read(&Min[i], sizeof(int));
      file->read(&Max[i], sizeof(int));
      file->read(&Avg[i], sizeof(int));
   }
   int len;
   file->read(&len, sizeof(len));
   char *buf = new char[len];
   file->read(buf, len);
   ItemName = buf;
   delete[] buf;
}

static std::vector<Item *> Items;
static int ItemIndex = 0, AttrIndex = 0;
static void SaveItemData();
static void ReadAttr();
static void SaveAttr();
static void ChangeItem();
static void ClearItems();
static void ReadAttr();

int LoadObjects()
{
   sbinfile file;
   char FileName[256], EnumFile[256];
   int i, num;

   FileName[0] = 0;
   g_Ofn.lpstrTitle = "Load Objects data";
   g_Ofn.lpstrFile  = FileName;
   g_Ofn.Flags      = OFN_FILEMUSTEXIST | OFN_HIDEREADONLY | OFN_PATHMUSTEXIST;

   if(!GetOpenFileName(&g_Ofn))
   {
      MessageBox(g_hWnd, "Item Editor cannot continue without loading an object file.", "Error!", MB_OK);
      return 1;
   }

   if (file.open(FileName, file.openro | file.openex))
   {
      MessageBox(g_hWnd, "Error opening object file", "Error!", MB_OK);
      return 2;
   }
   else
   {
      file.read(&num, sizeof(num));
      Objects.resize(num);
      for(i=0;i<num;i++)
      {
         Objects[i] = new Vectors::VectObject;
         Objects[i]->Load(&file);
      }
   }
   file.close();

   for(i=strlen(FileName);i>0;i--) if (FileName[i] == '\\') break;
   CopyMemory(EnumFile, FileName, i+1);
   EnumFile[i+1] = 0;
   strcat(EnumFile, "Objects.Enum");

   if (file.open(EnumFile, file.openro | file.openex))
   {
      MessageBox(g_hWnd, "Error opening enum file", "Error!", MB_OK);
      return 3;
   }
   else
   {
      int i, num = file.length() + 1;
      char *list = new char[num], *start, *end;
      HWND hwndCombo = GetDlgItem(g_hWnd, IDC_OBJCOMBO);
      start = list;
      list[num-1] = 0;
      file.read(list, num-1);

      num = Objects.size();

      for(i=0;i<num;i++)
      {
         int len;
         char temp[128];
         end = strchr(start, ',');
         if(end)
         {
            memcpy(temp, start, len = (end-start));
            temp[len] = 0;
            SendMessage(hwndCombo, CB_ADDSTRING, 0, (LPARAM) temp);
            start = end+2;
         }
         else SendMessage(hwndCombo, CB_ADDSTRING, 0, (LPARAM) start); 
         SendMessage(hwndCombo, CB_SETITEMDATA, i, (LPARAM) i);
      };
      delete list;
   }
   file.close();
   
   Items.resize(1);
   Items[0] = new Item;
   ChangeAttr(0);
   ChangeItem();

   return 0;
} /* LoadObjects */


int InitGraphics()
{
   HWND hwndObj = GetDlgItem(g_hWnd, IDC_OBJDISPLAY);
   RECT display;

   GetWindowRect(hwndObj, &display);
   DispWid = display.right  - display.left - 5;
   DispHei = display.bottom - display.top  - 19;
   UpdateCoords();

   if (Cddraw.Init(g_hWnd) != CDD_NOERROR) return 1;
   if (Cddraw.SetMode(DispWid, DispHei, CDD_16BIT, CDD_WINDOWED, CDD_SINGLE) != CDD_NOERROR) return 2;
   if (Cddraw.MakeSurface(DispWid, DispHei, CDD_16BIT, &BackBuffer, CDD_SYSTEM) != CDD_NOERROR) return 3;

   PD.SetCDDraw(&Cddraw);
   return 0;
} /* InitGraphics */


void UpdateCoords()
{
   RECT window, ctrl;
   GetClientRect(g_hWnd, &window);
   ClientToScreen(g_hWnd, (POINT*)&window.left);
   ClientToScreen(g_hWnd, (POINT*)&window.right);
   GetWindowRect(GetDlgItem(g_hWnd, IDC_OBJDISPLAY), &ctrl);
   DispX = ctrl.left - window.left + 2;
   DispY = ctrl.top  - window.top + 15;
   Cddraw.UpdateWindowed();
} /* UpdateCoords */


int RedrawObject()
{
   RECT dest;
   Cddraw.ClearSurface(BackBuffer);
   Objects[ObjIndex]->Draw(&PD, BackBuffer, DispWid/2, DispHei/2, 0, 0);

   dest.left   = DispX, dest.top = DispY;
   dest.right  = dest.left + DispWid;
   dest.bottom = dest.top  + DispHei;

   if (Cddraw.Blit(&Cddraw.Primary, BackBuffer, &dest, NULL) != CDD_NOERROR)
   {
      if (Cddraw.GetLastDDError() == DDERR_SURFACELOST)
      {
         if (Cddraw.RestoreAll() != CDD_NOERROR) return 1;
         if (Cddraw.Blit(&Cddraw.Primary, BackBuffer, &dest, NULL) != CDD_NOERROR) return 1;
      }
      else return 1;
   }
   return 0;
} /* RedrawObject */


void ChooseObject(int obj)
{
   ObjIndex = obj;
   SendMessage(GetDlgItem(g_hWnd, IDC_OBJCOMBO), CB_SETCURSEL, obj, 0);
   RedrawObject();
} /* ChooseObject */


void ChooseItem(int item)
{
   SaveItemData();
   ItemIndex = item;
   ChangeItem();
} /* ChooseItem */


void NewItem()
{
   SaveItemData();
   
   int num = Items.size();

   Items.resize(num+1);
   ItemIndex = num;
   Items[ItemIndex] = new Item;

   ChangeItem();
} /* NextItem */


void DelItem()
{
   int num = Items.size()-1;
   delete Items[ItemIndex];
   for(int i=ItemIndex;i<num;i++) Items[i] = Items[i+1];
   
   if (num)
   {
      Items.resize(num);
      SendMessage(GetDlgItem(g_hWnd, IDC_ITEMCOMBO), CB_DELETESTRING, ItemIndex, 0);
      if(ItemIndex == num) ItemIndex--;
   }
   else
   {
      Items[0] = new Item;
      SendMessage(GetDlgItem(g_hWnd, IDC_ITEMCOMBO), CB_DELETESTRING, 0, 0);
   }

   ChangeItem();
} /* DelItem */


void SaveItemData()
{
   char buf[128];

   GetWindowText(GetDlgItem(g_hWnd, IDE_ITEMNAME), buf, 128);
   Items[ItemIndex]->ItemName = buf;
      
   GetWindowText(GetDlgItem(g_hWnd, IDE_ENUMNAME), buf, 128);
   Items[ItemIndex]->EnumName = buf;
      
   GetWindowText(GetDlgItem(g_hWnd, IDE_CURSED), buf, 128);
   Items[ItemIndex]->Cursed = atoi(buf);

   GetWindowText(GetDlgItem(g_hWnd, IDE_WEIGHT), buf, 128);
   Items[ItemIndex]->Weight = atoi(buf);

   Items[ItemIndex]->Type   = SendMessage(GetDlgItem(g_hWnd, IDC_TYPECOMBO),  CB_GETCURSEL, 0, 0);
   Items[ItemIndex]->Object = SendMessage(GetDlgItem(g_hWnd, IDC_OBJCOMBO),   CB_GETCURSEL, 0, 0);
   Items[ItemIndex]->Place  = SendMessage(GetDlgItem(g_hWnd, IDC_EQUIPCOMBO), CB_GETCURSEL, 0, 0);
   
   SaveAttr();
} /* SaveItemData */


void ChangeItem()
{
   char buf[32];

   wsprintf(buf, "%d of %d", ItemIndex+1, Items.size());
   SetWindowText(GetDlgItem(g_hWnd, IDS_INDEX), buf);
   wsprintf(buf, "%d", Items[ItemIndex]->Cursed);
   SetWindowText(GetDlgItem(g_hWnd, IDE_CURSED), buf);
   wsprintf(buf, "%d", Items[ItemIndex]->Weight);
   SetWindowText(GetDlgItem(g_hWnd, IDE_WEIGHT), buf);
   SetWindowText(GetDlgItem(g_hWnd, IDE_ITEMNAME), Items[ItemIndex]->ItemName.c_str());
   SetWindowText(GetDlgItem(g_hWnd, IDE_ENUMNAME), Items[ItemIndex]->EnumName.c_str());
   UpdateItemBox();
   SendMessage(GetDlgItem(g_hWnd, IDC_TYPECOMBO),  CB_SETCURSEL, Items[ItemIndex]->Type,  0);
   SendMessage(GetDlgItem(g_hWnd, IDC_EQUIPCOMBO), CB_SETCURSEL, Items[ItemIndex]->Place, 0);
   SendMessage(GetDlgItem(g_hWnd, IDC_ITEMCOMBO),  CB_SETCURSEL, ItemIndex, 0);

   ReadAttr();
   
   int obj = Items[ItemIndex]->Object;
   if(ObjIndex != obj)  ChooseObject(obj);
} /* ChangeItem */


void UpdateItemBox()
{
   char buf[128];
   HWND hwndcombo = GetDlgItem(g_hWnd, IDC_ITEMCOMBO);

   GetWindowText(GetDlgItem(g_hWnd, IDE_ENUMNAME), buf, 128);
   SendMessage(hwndcombo, CB_DELETESTRING, ItemIndex, 0);
   SendMessage(hwndcombo, CB_INSERTSTRING, ItemIndex, (LPARAM)buf);
   SendMessage(hwndcombo, CB_SETCURSEL,    ItemIndex, 0);
} /* UpdateItemBox */


void ChangeAttr(int attr)
{
   SaveAttr();
   AttrIndex = attr;
   ReadAttr();
} /* ChangeAttr */


void ReadAttr()
{
   char buf[32];
   
   wsprintf(buf, "%d", Items[ItemIndex]->Min[AttrIndex]);
   SetWindowText(GetDlgItem(g_hWnd, IDE_MIN), buf);

   wsprintf(buf, "%d", Items[ItemIndex]->Max[AttrIndex]);
   SetWindowText(GetDlgItem(g_hWnd, IDE_MAX), buf);

   wsprintf(buf, "%d", Items[ItemIndex]->Avg[AttrIndex]);
   SetWindowText(GetDlgItem(g_hWnd, IDE_AVG), buf);
} /* ReadAttr */


void SaveAttr()
{
   char buf[32];

   GetWindowText(GetDlgItem(g_hWnd, IDE_MIN), buf, 32);
   Items[ItemIndex]->Min[AttrIndex] = atoi(buf);

   GetWindowText(GetDlgItem(g_hWnd, IDE_MAX), buf, 32);
   Items[ItemIndex]->Max[AttrIndex] = atoi(buf);

   GetWindowText(GetDlgItem(g_hWnd, IDE_AVG), buf, 32);
   Items[ItemIndex]->Avg[AttrIndex] = atoi(buf);
} /* SaveAttr */


void ClearItems()
{
   int i, num = Items.size();
   for(i=0;i<num;i++) delete Items[i];
   Items.resize(0);
   SendMessage(GetDlgItem(g_hWnd, IDC_ITEMCOMBO), CB_RESETCONTENT, 0, 0);
} /* ClearItems */


void SaveItems()
{
   char FileName[256], EnumName[256];
   sbinfile file, enumfile;
   int i, num = Items.size();
   char const *sep=", ", *name;

   FileName[0]      = 0;
   g_Ofn.Flags      = OFN_CREATEPROMPT    | OFN_HIDEREADONLY | OFN_NOREADONLYRETURN |
                      OFN_OVERWRITEPROMPT | OFN_PATHMUSTEXIST;
   g_Ofn.lpstrFile  = FileName;
   g_Ofn.lpstrTitle = "Save Item Data";

   if(!GetSaveFileName(&g_Ofn))
   {
      MessageBox(g_hWnd, "Data not saved.", "Warning!", MB_OK);
      return;
   }

   for(i=strlen(FileName);i>0;i--) if (FileName[i] == '\\') break;
   CopyMemory(EnumName, FileName, i+1);
   EnumName[i+1] = 0;
   strcat(EnumName, "Items.Enum");

   if(file.open(FileName, file.openrw | file.opentr) ||
      enumfile.open(EnumName, file.openrw | file.opentr))
   {
      MessageBox(g_hWnd, "Error opening save file", "Error!", MB_OK);
      return;
   }
   
   SaveItemData();
   
   file.write(&num, sizeof(int));
   for(i=0;i<num;i++)
   {
      Items[i]->Save(&file);
      name = Items[i]->EnumName.c_str();
      enumfile.write(name, strlen(name));
      enumfile.write(sep, 2);
   }

   file.close();
   enumfile.close();
} /* SaveItems */


void LoadItems()
{
   HWND hwndcombo;
   char FileName[256], EnumName[256], temp[128];
   sbinfile file, enumfile;
   int i, num, len;
   char *list, *start, *end;

   FileName[0]      = 0;
   g_Ofn.Flags      = OFN_FILEMUSTEXIST | OFN_HIDEREADONLY | OFN_PATHMUSTEXIST;
   g_Ofn.lpstrFile  = FileName;
   g_Ofn.lpstrTitle = "Load Item Data";

   if(!GetOpenFileName(&g_Ofn)) return;

   for(i=strlen(FileName);i>0;i--) if (FileName[i] == '\\') break;
   CopyMemory(EnumName, FileName, i+1);
   EnumName[i+1] = 0;
   strcat(EnumName, "Items.Enum");

   if(file.open(FileName, file.openro) || enumfile.open(EnumName, file.openro))
   {
      MessageBox(g_hWnd, "Error opening save file", "Error!", MB_OK);
      return;
   }
   
   ClearItems();

   len = enumfile.length();
   list = new char[len+1];
   enumfile.read(list, len);
   list[len] = 0;
   start = list;

   hwndcombo = GetDlgItem(g_hWnd, IDC_ITEMCOMBO);

   file.read(&num, sizeof(num));
   Items.resize(num);

   for(i=0;i<num;i++)
   {
      Items[i] = new Item;
      Items[i]->Load(&file);
      end = strchr(start, ',');
      if(end)
      {
         len = end-start;
         memcpy(temp, start, len);
         temp[len] = 0;
         Items[i]->EnumName = temp;
         start = end+2;
      }
      else Items[i]->EnumName = start;
      SendMessage(hwndcombo, CB_ADDSTRING, 0, (LPARAM)Items[i]->EnumName.c_str());
   }
   delete[] list;
   
   file.close();
   enumfile.close();

   ItemIndex = 0;
   ChangeItem();
} /* LoadItems */
